const db = require("../config/db");

const createBlog = async (req, res) => {
  try {
    const {
      name,
      slug,
      description,
      media_ids,
      meta_title,
      meta_description,
      category_ids,
    } = req.body;

    // Validate required fields
    if (!name) {
      return res.status(400).json({
        success: false,
        message: "Name is required.",
      });
    }

    // 2️ Ensure category_ids is an array and convert to comma-separated string safely
    let categoryIdStr = "";
    if (Array.isArray(category_ids) && category_ids.length > 0) {
      categoryIdStr = category_ids.join(",");
    } else if (typeof category_ids === "string" && category_ids.trim() !== "") {
      categoryIdStr = category_ids.trim();
    }

    // First, check if the slug already exists in the 'categories' table
    const [categorySlugCheck] = await db.execute(
      `SELECT * FROM blogs WHERE slug = ?`,
      [slug],
    );

    if (categorySlugCheck.length > 0) {
      return res.status(400).json({
        success: false,
        message: "Slug already exists.",
      });
    }

    // Handle optional media_ids safely (convert array to string)
    let mediaIdsAsString = null;
    if (Array.isArray(media_ids) && media_ids.length > 0) {
      mediaIdsAsString = media_ids.join(","); // e.g. "1,2,3"
    }

    // Debug log (optional)
    // console.log("Create Blog Params:", {
    //   name,
    //   description,
    //   mediaIdsAsString,
    // });

    const nowUtc = new Date().toISOString().slice(0, 19).replace("T", " ");

    // Insert into the blogs table
    const [result] = await db.execute(
      `INSERT INTO blogs (name, description, media_ids,meta_title,meta_description,created_at_utc,slug,category_ids)
       VALUES (?, ?, ?,?,?,?,?,?)`,
      [
        name,
        description,
        mediaIdsAsString,
        meta_title,
        meta_description,
        nowUtc,
        slug,
        categoryIdStr,
      ],
    );

    // Respond with success
    if (result.affectedRows > 0) {
      return res.status(201).json({
        success: true,
        message: "Blog created successfully.",
        blogId: result.insertId,
      });
    } else {
      return res.status(500).json({
        success: false,
        message: "Failed to create blog.",
      });
    }
  } catch (error) {
    console.error("Create Blog Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
      error: error.message,
    });
  }
};

function normalizeSlug(slug) {
  return slug
    .toLowerCase()
    .replace(/[^a-z0-9]/g, "-")
    .replace(/-{2,}/g, "-")
    .replace(/^-+|-+$/g, "");
}

const checkSlugExist = async (req, res) => {
  try {
    const { slug } = req.params;

    // console.log("Slug received:", slug);
    // console.log("Slug type:", typeof slug);

    // agar normalizeSlug function use kar rahe hain to sahi call karen:
    const normalizedSlug = normalizeSlug(slug);

    // console.log("Normalized Slug:", normalizedSlug);
    // console.log("Normalized Slug type:", typeof normalizedSlug);

    const [result] = await db.execute("SELECT * FROM blogs WHERE slug = ?", [
      normalizedSlug,
    ]);

    if (result.length > 0) {
      return res.json({ success: false, message: "Slug already exist." });
    }
    return res.json({ success: true, message: "Slug is available." });
  } catch (error) {
    console.error("Check slug exist Error:", error);
    return res.status(500).json({ success: false, message: "Server error." });
  }
};

const getBlogById = async (req, res) => {
  try {
    const { id } = req.params;

    // Validate blog ID
    if (!id || isNaN(id)) {
      return res.status(400).json({
        success: false,
        message: "Invalid blog ID.",
      });
    }

    // Fetch the blog by ID
    const [rows] = await db.execute(
      `SELECT * FROM blogs WHERE id = ? LIMIT 1`,
      [id],
    );

    // If not found
    if (rows.length === 0) {
      return res.status(404).json({
        success: false,
        message: "Blog not found.",
      });
    }

    // Respond with blog data
    return res.status(200).json({
      success: true,
      data: rows[0],
    });
  } catch (error) {
    console.error("Get Blog Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
      error: error.message,
    });
  }
};

const getAllBlogs = async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 100;
    const offset = (page - 1) * limit;

    // Total count
    const [countResult] = await db.query("SELECT COUNT(*) AS total FROM blogs");
    const total = countResult[0].total;

    // Fetch blogs with pagination
    const query = `SELECT * FROM blogs ORDER BY created_at_utc DESC LIMIT ? OFFSET ?`;
    const [result] = await db.execute(query, [limit, offset]);

    // If no blogs found
    if (result.length === 0) {
      return res.status(404).json({
        success: false,
        message: "No blogs found.",
      });
    }

    // Return blogs with pagination info
    return res.json({
      success: true,
      blogs: result,
      currentPage: page,
      totalPages: Math.ceil(total / limit),
      totalRecords: total,
    });
  } catch (error) {
    console.error("Get Blogs Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
    });
  }
};

const updateBlog = async (req, res) => {
  try {
    const { id } = req.params;
    const {
      name,
      description,
      media_ids,
      meta_title,
      meta_description,
      slug,
      category_ids,
    } = req.body;

    // Validate ID
    if (!id || isNaN(id)) {
      return res.status(400).json({
        success: false,
        message: "Invalid blog ID.",
      });
    }

    // Check if blog exists
    const [existing] = await db.execute(`SELECT * FROM blogs WHERE id = ?`, [
      id,
    ]);
    if (existing.length === 0) {
      return res.status(404).json({
        success: false,
        message: "Blog not found.",
      });
    }

    // Prepare media_ids string
    let mediaIdsAsString = null;
    if (Array.isArray(media_ids) && media_ids.length > 0) {
      mediaIdsAsString = media_ids.join(","); // e.g. "1,2,3"
    }

    // Normalize category_ids to comma-separated string
    let categoryIdStr = "";
    if (Array.isArray(category_ids) && category_ids.length > 0) {
      categoryIdStr = category_ids.join(",");
    } else if (typeof category_ids === "string" && category_ids.trim() !== "") {
      categoryIdStr = category_ids.trim();
    }

    // Update query
    const [result] = await db.execute(
      `UPDATE blogs SET name = ?, description = ?, media_ids = ? ,meta_title = ? , meta_description = ? , slug = ? , category_ids = ? WHERE id = ?`,
      [
        name ?? existing[0].name,
        description ?? existing[0].description,
        mediaIdsAsString ?? existing[0].media_ids,
        meta_title ?? existing[0].meta_title,
        meta_description ?? existing[0].meta_description,
        slug ?? existing[0].slug,
        categoryIdStr ?? existing[0].category_ids,
        id,
      ],
    );

    return res.json({
      success: true,
      message: "Blog updated successfully.",
    });
  } catch (error) {
    console.error("Update Blog Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
      error: error.message,
    });
  }
};

const deleteBlog = async (req, res) => {
  try {
    const { id } = req.params;

    // Validate blog ID
    if (!id || isNaN(id)) {
      return res.status(400).json({
        success: false,
        message: "Invalid blog ID.",
      });
    }

    // Check if blog exists
    const [existing] = await db.execute(`SELECT * FROM blogs WHERE id = ?`, [
      id,
    ]);
    if (existing.length === 0) {
      return res.status(404).json({
        success: false,
        message: "Blog not found.",
      });
    }

    // Delete the blog
    const [result] = await db.execute(`DELETE FROM blogs WHERE id = ?`, [id]);

    if (result.affectedRows > 0) {
      return res.json({
        success: true,
        message: "Blog deleted successfully.",
      });
    } else {
      return res.status(500).json({
        success: false,
        message: "Failed to delete blog.",
      });
    }
  } catch (error) {
    console.error("Delete Blog Error:", error);
    return res.status(500).json({
      success: false,
      message: "Server error.",
      error: error.message,
    });
  }
};

const changeBlogStatus = async (req, res) => {
  try {
    const blogId = parseInt(req.params.blogId, 10);
    const { status } = req.body;

    // console.log(commentId,status)

    // Validate input: status can be "0" or "1", so check for undefined/null
    if (!blogId || status === undefined || status === null) {
      return res.status(400).json({ message: "Missing required fields" });
    }

    // console.log(commentId);

    const query = `UPDATE blogs SET is_active_or_inactive = ? WHERE id = ?`;
    const params = [status, blogId];

    const [result] = await db.query(query, params);

    if (result.affectedRows > 0) {
      return res.json({
        success: true,
        message: "Status changed successfully.",
      });
    }

    return res.status(404).json({
      success: false,
      message: "No record found to update.",
    });
  } catch (err) {
    console.error("Error in change CommentStatus:", err);
    return res.status(500).json({
      success: false,
      message: "An error occurred while changing status.",
    });
  }
};

module.exports = {
  createBlog,
  getAllBlogs,
  getBlogById,
  deleteBlog,
  updateBlog,
  changeBlogStatus,
  checkSlugExist,
};
